import 'dart:convert';

import 'package:flutter/material.dart';
import 'package:flutter_ringtone_player/flutter_ringtone_player.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import 'package:pusher_channels_flutter/pusher_channels_flutter.dart';
import 'package:razinsoft_rider/core/routes/app_routes.dart';
import 'package:razinsoft_rider/core/utils/helpers.dart';
import 'package:razinsoft_rider/core/widgets/is_ios.dart';
import 'package:razinsoft_rider/data/models/chat_message_response/chat_message.dart';
import 'package:razinsoft_rider/data/models/travel_info_model/travel_info_model.dart';
import 'package:razinsoft_rider/data/services/local_storage_service.dart';
import 'package:razinsoft_rider/data/services/navigation_service.dart';
import 'package:razinsoft_rider/data/services/pusher_service.dart';
import 'package:razinsoft_rider/presentation/booking/provider/booking_providers.dart';
import 'package:razinsoft_rider/presentation/booking/provider/order_providers.dart';
import 'package:razinsoft_rider/presentation/booking/provider/pusher_provider.dart';
import 'package:razinsoft_rider/presentation/booking/provider/route_providers.dart';
import 'package:razinsoft_rider/presentation/track_order/provider/chat_provider.dart';
import 'package:razinsoft_rider/presentation/waypoint/provider/way_point_map_providers.dart';
import 'package:vibration/vibration.dart';

import '../../../data/models/order_response/order_model/order/order.dart';
import '../../track_order/view_model/handle_order_status_update.dart';

class PusherNotifier extends StateNotifier<void> {
  final Ref ref;
  PusherNotifier(this.ref) : super(null);

  void subscribeAllChannels(){
      final Order? order = ref.read(createOrderNotifierProvider).whenOrNull(success: (data)=> data) ;
      if(order == null) return;
      final riderId = order.rider?.id;
      final riderChannelSuffix = '${order.id}.$riderId';
      PusherService().subscribeChannel('order.$riderChannelSuffix');
      PusherService().subscribeChannel('status.$riderChannelSuffix');
      PusherService().subscribeChannel('chat_$riderId');
      PusherService().subscribeChannel('order-decline.$riderId');
      PusherService().subscribeChannel('driver-current-location.$riderId');
      PusherService().subscribeChannel('order-cancelled.$riderId');
      PusherService().subscribeChannel('order-cash.$riderId');
  }

  void unsubscribeAll(){
    PusherService().unsubscribeAllChannels();
  }

  Future<void> setupPusherListeners({bool connectAllChannels = false}) async {
    await PusherService().init(onEvent: _onPusherEventReceived);
    if(connectAllChannels){
      subscribeAllChannels();
    }
  }

  Future<void> _onPusherEventReceived(PusherEvent event) async {

    final data = _parseEventData(event.data);

    if (data.isEmpty) {
      return;
    }

    handleOrderOrStatusEvent(event, data);

  }

  Map<String, dynamic> _parseEventData(dynamic data) {
    try {
      if (data == null) return {};
      if (data is String) return Map<String, dynamic>.from(jsonDecode(data));
      if (data is Map) return Map<String, dynamic>.from(data);
    } catch (e) {
      if(true){}
    }
    return {};
  }

  void handleOrderOrStatusEvent(PusherEvent event, Map<String, dynamic> data, {bool fromLocal = false}) async{
    final local = LocalStorageService();
    final int? orderId = await local.getOrderId();
    final int riderId = await local.getUserId();

    if (orderId == null || riderId == 0) return;

    final orderChannel = 'order.$orderId.$riderId';
    final locationUpdateChannel = 'driver-current-location.$riderId';

    if (event.channelName.contains(orderChannel)) {
      _handleOrderChannel(event, data, locationUpdateChannel);
    } else if (event.channelName.contains(locationUpdateChannel)) {
      if(event.eventName.toLowerCase().contains('send-travel-info-notification')){
        _handleTravelInfoUpdate(data);
      }else if(event.eventName.toLowerCase().contains('driver-current-location-notification')){
        _handleLocationUpdate(data);
      }else{
      }

    }else if(event.channelName.contains('order-cash.$riderId')){
      playRingtone();
      showNotification(message: 'Your Ride Has Been Completed!', isSuccess: true);
      ref.read(bookingNotifierProvider.notifier).resetState();
      ref.read(pusherNotifierProvider.notifier).disconnect();
      LocalStorageService().clearOrderId();
      NavigationService.pushNamedAndRemoveUntil(AppRoutes.dashboard);
    }else if(true){
      await _handleChatEvent(event, data);
    }
  }

  void _handleOrderChannel(PusherEvent event, Map<String, dynamic> data, String locationUpdate) {
    playRingtone();

    switch (event.eventName.toLowerCase()) {
      case 'cancelled-order-notification':
        final msg = data['message'] ?? 'Your Ride Has Been Declined! Please try again later.';
        showNotification(message: msg);
        ref.read(bookingNotifierProvider.notifier).resetState();
        ref.read(pusherNotifierProvider.notifier).disconnect();
        LocalStorageService().clearOrderId();
        NavigationService.pushNamedAndRemoveUntil(AppRoutes.dashboard);
        break;

      case 'status-update-notification':
        _handleInnerStatusData(data['data'], locationUpdate);
        break;

      case 'declined-looking-driver':
        showNotification(message: 'Driver canceled your ride!. Please wait we are looking for driver.');
        ref.read(createOrderNotifierProvider).whenOrNull(success: (order)=> ref.read(createOrderNotifierProvider.notifier).handleOrderCreationSuccess(order));
        break;
    }
  }

  void _handleInnerStatusData(dynamic innerData, String locationUpdate) {
    if (innerData is! Map || !innerData.containsKey('status')) return;

    final status = innerData['status']?.toLowerCase();
    final orderId = innerData['order_id'];
    if (status == null) return;
    handleOrderStatusUpdate( status: status, orderId: orderId, ref: ref, fromPusher: true);
  }

  void _handleLocationUpdate(Map<String, dynamic> data) {
    final inner = data['data'];
    if (inner == null) return;
    

    final lat = double.tryParse(inner['lat'].toString());
    final lng = double.tryParse(inner['lng'].toString());

    if (lat != null && lng != null) {
      ref.read(wayPointMapNotifierProvider.notifier).updateForGoToPickup(LatLng(lat, lng));
    }
  }
  void _handleTravelInfoUpdate(Map<String, dynamic> data){
    try{
      final TravelInfoModel info = TravelInfoModel.fromJson(data);
      ref.read(routeProgressProvider.notifier).state = info.data?.progress?.toDouble() ?? 0.0;
      ref.read(travelInfoNotifierProvider.notifier).setTimeAndDistance(time: info.data?.minute, distance: info.data?.distance);

    }catch (e){
      if(true){}
    }
  }

  Future<void> _handleChatEvent(PusherEvent event, Map<String, dynamic> data) async {
    final userId = await LocalStorageService().getUserId();
    final chatChannel = 'chat_$userId';

    if (event.channelName.contains(chatChannel)) {
      _processChatMessage(data);
    }
  }

  void _processChatMessage(Map<String, dynamic> data) async {
    final message = Message.fromJson(data, convertPusher: true);
    final isChatOpen = await LocalStorageService().getChatState();

    if (isChatOpen) {
      ref.read(chatNotifierProvider.notifier).addMessage(message);
    }else{
      playRingtone();
      _showChatSnackBar(message);
    }
  }

  /// Show snack bar for new chat message
  void _showChatSnackBar(Message message) {
    final context = NavigationService.navigatorKey.currentContext;
    if (context == null) return;

    const sender = 'New Message from Driver';
    final msgText = message.message;

    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(
        content: RichText(
          maxLines: 3,
          overflow: TextOverflow.ellipsis,
          text: TextSpan(
            style: const TextStyle(color: Colors.white),

            children: [
              const TextSpan(
                text: '$sender: ',

                style: TextStyle(fontWeight: FontWeight.bold,),
              ),
              TextSpan(text: msgText),
            ],
          ),
        ),
        action: SnackBarAction(
          label: 'See',

          onPressed: () async{
            // ref.read(trackOrderNotifierProvider.notifier).goToChat();
            if(await LocalStorageService().getChatState())return;
            NavigationService.pushNamed(AppRoutes.chatPage);
          },
          textColor: Colors.yellowAccent,

        ),
        backgroundColor: Colors.green,
        duration: const Duration(seconds: 6),
        behavior: SnackBarBehavior.floating,
        margin: const EdgeInsets.symmetric(horizontal: 12, vertical: 10),
        shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(8)),
      ),
    );
  }

  Future<void> disconnect() async {
    await PusherService().disconnectAndUnsubscribeAll();
  }

  Future<void> reset() async {
    await PusherService().reset();
  }

  void playRingtone([bool off = false]) {
    if (!off) {
      if(isIos()){
        FlutterRingtonePlayer().play(
          android: AndroidSounds.notification,
          ios: IosSounds.horn,
          looping: false,
          volume: 1.0,
        );
      }else{
        FlutterRingtonePlayer().play(
          fromAsset: 'assets/notification2.mp3',
          looping: false,
          volume: 1.0,
          asAlarm: true,
        );
      }

    }
    _vibratePhone(off);
  }

  Future<void> _vibratePhone([bool off = false]) async {
    if (off) return;

    final enabled = await LocalStorageService().getVibration();
    if (!enabled) return;

    if (await Vibration.hasCustomVibrationsSupport()) {
      Vibration.vibrate(duration: 2000);
    } else {
      Vibration.vibrate();
      await Future.delayed(const Duration(milliseconds: 500));
      Vibration.vibrate();
    }
  }
}
